﻿using System;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Collections.Generic;
using Newtonsoft.Json;
using Newtonsoft.Json.Serialization;
using System.Text.Json.Nodes;
using Shared.Models.Module;
using Telegram.Bot;
using Telegram.Bot.Exceptions;
using Telegram.Bot.Types;
using Telegram.Bot.Types.Enums;
using Telegram.Bot.Types.ReplyMarkups;
using Telegram.Bot.Polling;

namespace TelegramBot
{
    public class ModInit
    {
        private static ITelegramBotClient _bot;
        private static readonly string userFilePath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "users.json");
        private static readonly string stateFilePath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "module", "my-TelegramBot", "stateFile.json");
        private static string curid = null;
        private static CancellationTokenSource _cts;
        private static long adminId = 0;

        public static void loaded(InitspaceModel conf)
        {
            ThreadPool.QueueUserWorkItem(async _ =>
            {
                try
                {
                    var manifestPath = Path.Combine(conf.path, "manifest.json");
                    var manifestJson = File.ReadAllText(manifestPath);
                    var p = JsonNode.Parse(manifestJson)["params"];
                    string bottoken = p["bottoken"].GetValue<string>();
                    adminId = long.Parse(p["adminid"].ToString());
                    _cts = new CancellationTokenSource();
                    _bot = new TelegramBotClient(bottoken, cancellationToken: _cts.Token);

                    _bot.StartReceiving(
                        new DefaultUpdateHandler(UpdateHandler, ErrorHandler),
                        new ReceiverOptions
                        {
                            AllowedUpdates = new[] { UpdateType.Message, UpdateType.CallbackQuery }
                        },
                        _cts.Token
                    );

                    var me = await _bot.GetMeAsync();
                    Console.WriteLine($"\n\t@{me.Username} is running...\n\n");

                    await Task.Delay(-1, _cts.Token);
                }
                catch (Exception ex) { Console.WriteLine("TelegramBot Exception: " + ex + "\n\n"); }
            });
        }

        private static async Task UpdateHandler(ITelegramBotClient bot, Update update, CancellationToken cancellationToken)
        {
            if (update.Type == UpdateType.Message && update.Message.Type == MessageType.Text)
            {
                await HandleMessage(bot, update.Message, cancellationToken);
            }
            else if (update.Type == UpdateType.CallbackQuery)
            {
                await CallbackQueryHandler(bot, update.CallbackQuery, cancellationToken);
            }
        }

        private static async Task HandleMessage(ITelegramBotClient bot, Message message, CancellationToken cancellationToken)
        {
            var chatId = message.Chat.Id;
            bool admin = chatId == adminId;
            string response = "";
            var users = LoadUsers();
            ReplyMarkup replyMarkup = null;
            var states = LoadState() ?? new List<StateInfo>();
            var stateinfo = states.FirstOrDefault(s => s.chatId == chatId);
            if (stateinfo == null)
            {
                stateinfo = new StateInfo { chatId = chatId, state = "start" };
                states.Add(stateinfo);
            }
            response = await HandleNextMessage(admin, bot, message, stateinfo, users, cancellationToken);
            replyMarkup = GetReplyMarkup(stateinfo.state, admin, stateinfo.id);
            if (response != null) await bot.SendTextMessageAsync(chatId, response, replyMarkup: replyMarkup, cancellationToken: cancellationToken);
            SaveState(states);
        }

        private static async Task<string> HandleNextMessage(bool admin, ITelegramBotClient bot, Message message, StateInfo stateinfo, List<User> users, CancellationToken cancellationToken)
        {
            string response = "";
            switch (stateinfo.state)
            {
                case "start":    /// Запуск
                    if (admin)
                    {
                        response = await HandleAdminStartState(bot, message, stateinfo, users, cancellationToken);
                    }
                    else
                    {
                        response = await HandleUserStartState(bot, message, stateinfo, users, cancellationToken);
                    }
                    break;
                case "newUser":    /// Добавление нового пользователя
                    response = HandleNewUserState(message, stateinfo, users, admin, bot);
                    break;
                case "addcomment":  /// Добавление комментария
                    response = HandleAddCommentState(message, stateinfo, users);
                    break;
                case "addids":      /// Вопрос про ID
                    response = HandleAddIdsState(message, stateinfo);
                    break;
                case "addid":       /// Добавление ID
                    response = HandleAddIdState(message, stateinfo, users);
                    break;
            }
            return response;
        }

        private static async Task<string> HandleAdminStartState(ITelegramBotClient bot, Message message, StateInfo stateinfo, List<User> users, CancellationToken cancellationToken)
        {
            string response = "";
            switch (message.Text.ToLower())
            {
                case "/start":
                    response = "Привет, админ!";
                    break;
                case "новый пользователь":
                    stateinfo.state = "newUser";
                    response = "Введите ID пользователя:";
                    break;
                case "удалить пользователя":
                    response = await HandleDeleteUser(bot, message.Chat.Id, users, cancellationToken);
                    break;
                case "пользователи":
                    response = await HandleListUsers(bot, message.Chat.Id, users, cancellationToken);
                    break;
                case "добавить id":
                    response = await HandleAddIds(bot, message.Chat.Id, users, cancellationToken);
                    break; 
                case "изменить параметры":
                    response = await HandleChangeUserParams(bot, message.Chat.Id, users, cancellationToken);
                    break;
                case "тест":
                    response = $"@{message.From.Username} {message.From.Id}";
                    break;
                case "продлить доступ":
                    response = await HandleExtendService(bot, message.Chat.Id, users, cancellationToken);
                    break;
                default:
                    response = "Мимо =)";
                    break;
            }
            return response;
        }

        private static async Task<string> HandleUserStartState(ITelegramBotClient bot, Message message, StateInfo stateinfo, List<User> users, CancellationToken cancellationToken)
        {
            string response = "";
            switch (message.Text.ToLower())
            {
                case "/start":
                    response = "Привет!";
                    break;
                case "новый пользователь":
                    stateinfo.state = "newUser";
                    response = "Введите ID пользователя:";
                    break;
                case "добавить id":
                    ///response = await HandleAddIds(bot, message.Chat.Id, users, cancellationToken);
                    response = "Введите дополнительный ID:";
                    stateinfo.state = "addid";
                    break;
                case "техподдержка":
                    response = "Соберись, тряпка, ты сможешь!";
                    break;
                default:
                    response = "Выберите один из вариантов";
                    break;
            }
            return response;
        }

        private static string HandleNewUserState(Message message, StateInfo stateinfo, List<User> users, bool admin, ITelegramBotClient bot)
        {
            string response = "";
            stateinfo.id = message.Text;
            if (message.Text == "Отмена")
            {
                response = "Добавление нового пользователя отменено";
                stateinfo.state = "start";
            }
            else if (users.Any(u => u.id == message.Text))
            {
                response = "ID уже существует, пожалуйста, выберите другой ID:";
            }
            else
            {
                if (admin)
                {
                    AddUser(message.Text, _expires: DateTime.Now.AddMonths(1).ToString("yyyy-MM-ddTHH:mm:sszzz"));
                    response = "Пользователь успешно добавлен!\nВведите комментарий:";
                    stateinfo.state = "addcomment";
                }
                else
                {
                    AddUser(message.Text, _chatId: message.Chat.Id, _registration: DateTime.Now.ToString("yyyy-MM-ddTHH:mm:ss"));
                    response = "Пользователь успешно добавлен!\nВвести дополнительный ID?";
                    bot.SendTextMessageAsync(adminId, $"Новый пользователь {message.Text}");
                    stateinfo.state = "addids";
                }

            }
            return response;
        }

        private static string HandleAddCommentState(Message message, StateInfo stateinfo, List<User> users)
        {
            string response = "";
            var user = users.FirstOrDefault(u => u.id == stateinfo.id);
            if (user != null)
            {
                user.comment = message.Text;
                SaveUsers(users);
                response = "Комментарий успешно добавлен!\nВвести дополнительный ID?";
                stateinfo.state = "addids";
            }
            else
            {
                response = "Пользователь не найден";
                stateinfo.state = "start";
            }
            return response;
        }

        private static string HandleAddIdsState(Message message, StateInfo stateinfo)
        {
            string response = "";
            switch (message.Text.ToLower())
            {
                case "да":
                    response = "Введите дополнительный ID:";
                    stateinfo.state = "addid";
                    break;
                case "нет":
                    response = "Спасибо!";
                    stateinfo.state = "start";
                    break;
                default:
                    response = "Выберите один из вариантов";
                    break;
            }
            return response;
        }

        private static string HandleAddIdState(Message message, StateInfo stateinfo, List<User> users)
        {
            string response = "";
            if (message.Text == "Отмена")
            {
                response = "Добавление дополнительного ID отменено";
                stateinfo.state = "start";
            }
            var user = users.FirstOrDefault(u => u.id == stateinfo.id);
            if (user != null)
            {
                if (user.ids.Contains(message.Text))
                {
                    response = "ID уже существует, пожалуйста, выберите другой ID:";
                }
                else
                {
                    user.ids.Add(message.Text.ToLower());
                    SaveUsers(users);
                    response = "Дополнительный ID успешно добавлен! Добавить еще один ID?";
                    stateinfo.state = "addids";
                }
            }
            else
            {
                response = "Пользователь не найден";
                stateinfo.state = "start";
            }
            return response;
        }
        /// <summary>
        /// Видалення користувача
        /// </summary>

        
        private static async Task<string> HandleDeleteUser(ITelegramBotClient bot, long chatId, List<User> users, CancellationToken cancellationToken)
        {
            if (users.Count == 0)
            {
                return "Список пользователей пуст.";
            }
            foreach (var user in users)
            {
                var inlineKeyboard = new InlineKeyboardMarkup(new[]
                {
                    new[]
                    {
                        InlineKeyboardButton.WithCallbackData("🗑 Удалить", $"delete:{user.id}")
                    }
                });
                await bot.SendTextMessageAsync(chatId, $"{user.id}", replyMarkup: inlineKeyboard, cancellationToken: cancellationToken);
            }
            return null; // Щоб не відправляти зайве повідомлення
        }
        /// <summary>
        /// Вивід списку користувачів
        /// </summary>
        private static async Task<string> HandleListUsers(ITelegramBotClient bot, long chatId, List<User> users, CancellationToken cancellationToken)
        {
            if (users.Count == 0)
            {
                return "Список пользователей пуст.";
            }
            foreach (var user in users)
            {
                var inlineKeyboard = new InlineKeyboardMarkup(new[]
                {
                    new[]
                    {
                        InlineKeyboardButton.WithCallbackData("Инфо", $"info:{user.id}")
                    }
                });
                await bot.SendTextMessageAsync(chatId, $"{user.id}", replyMarkup: inlineKeyboard, cancellationToken: cancellationToken);
            }
            return null; // Щоб не відправляти зайве повідомлення
        }
        /// <summary>
        /// Добавлення ID через список
        /// </summary>
        private static async Task<string> HandleAddIds(ITelegramBotClient bot, long chatId, List<User> users, CancellationToken cancellationToken)
        {
            if (users.Count == 0)
            {
                return "Список пользователей пуст.";
            }
            foreach (var user in users)
            {
                var inlineKeyboard = new InlineKeyboardMarkup(new[]
                {
                    new[]
                    {
                        InlineKeyboardButton.WithCallbackData("Добавить ids", $"addids:{user.id}")
                    }
                });
                await bot.SendTextMessageAsync(chatId, $"{user.id}\n{user.comment}\nIDs: {string.Join(", ", user.ids)}", replyMarkup: inlineKeyboard, cancellationToken: cancellationToken);
            }
            return null; // Щоб не відправляти зайве повідомлення
        }

        private static async Task<string> HandleExtendService(ITelegramBotClient bot, long chatId, List<User> users, CancellationToken cancellationToken)
        {
            if (users.Count == 0)
            {
                return "Список пользователей пуст.";
            }
            foreach (var user in users)
            {
                var inlineKeyboard = new InlineKeyboardMarkup(new[]
                {
                    new[]
                    {
                        InlineKeyboardButton.WithCallbackData("Месяц+", $"extendservice:{user.id}")
                    }
                });
                await bot.SendTextMessageAsync(chatId, $"{user.id}\nДо {DateTime.Parse(user.expires).ToString("dd.MM.yyyy")}\nIDs: {string.Join(", ", user.ids)}", replyMarkup: inlineKeyboard, cancellationToken: cancellationToken);
            }
            return null; // Щоб не відправляти зайве повідомлення
        }

        private static async Task<string> HandleChangeUserParams(ITelegramBotClient bot, long chatId, List<User> users, CancellationToken cancellationToken)
        {
            if (users.Count == 0)
            {
                return "Список пользователей пуст.";
            }
            foreach (var user in users)
            {
                var inlineKeyboard = new InlineKeyboardMarkup(new[]
                {
                    new[]
                    {
                        InlineKeyboardButton.WithCallbackData("Редактировать параметры", $"editparams:{user.id}")
                    }
                });
                await bot.SendTextMessageAsync(chatId, $"{user.id}", replyMarkup: inlineKeyboard, cancellationToken: cancellationToken);
            }
            return null; // Щоб не відправляти зайве повідомлення
        }
        /// <summary>
        /// Вибір клавіатури в залежності від стану
        /// </summary>
        private static ReplyMarkup GetReplyMarkup(string state, bool admin, string id)
        {
            switch (state)
            {
                case "start":
                    if (admin)
                    {
                        return new ReplyKeyboardMarkup(new[]
                        {
                            new KeyboardButton[] { "Новый пользователь", "Пользователи" },
                            new KeyboardButton[] { "Добавить ID", "Удалить пользователя" },
                            new KeyboardButton[] { "Изменить параметры", "Продлить доступ" }
                        })
                        {
                            ResizeKeyboard = true
                        };
                    }
                    else if (!admin && id != null)
                    {
                        return new ReplyKeyboardMarkup(new[]
                        {
                            new KeyboardButton[] { "Добавить ID", "Техподдержка" }
                        })
                        {
                            ResizeKeyboard = true
                        };
                    }
                    else
                    {
                        return new ReplyKeyboardMarkup(new[]
                        {
                            new KeyboardButton[] { "Новый пользователь", "Добавить ID" },
                            new KeyboardButton[] { "Техподдержка" }
                        })
                        {
                            ResizeKeyboard = true
                        };
                    }
                case "addids":
                    return new ReplyKeyboardMarkup(new[]
                    {
                        new KeyboardButton[] { "Да", "Нет" }
                    })
                    {
                        ResizeKeyboard = true
                    };
                case "addid":
                    return new ReplyKeyboardMarkup(new[]
                    {
                        new KeyboardButton[] { "Отмена" }
                    })
                    {
                        ResizeKeyboard = true
                    };
                case "newUser":
                    return new ReplyKeyboardMarkup(new[]
                    {
                        new KeyboardButton[] { "Отмена" }
                    })
                    {
                        ResizeKeyboard = true
                    };
                default:
                    return new ReplyKeyboardRemove();
            }
        }

        private static async Task CallbackQueryHandler(ITelegramBotClient bot, CallbackQuery callbackQuery, CancellationToken cancellationToken)
        {
            var chatId = callbackQuery.Message.Chat.Id;
            var data = callbackQuery.Data.Split(':');

            if (data.Length < 2)
                return;

            var action = data[0];
            var userId = data[1];

            if (action == "delete")
            {
                var users = LoadUsers();
                var user = users.FirstOrDefault(u => u.id == userId);

                if (user != null)
                {
                    users.Remove(user);
                    SaveUsers(users);
                    await bot.EditMessageTextAsync(chatId, callbackQuery.Message.MessageId, $"❌ Пользователь {userId} удален.", cancellationToken: cancellationToken);
                }
                else
                {
                    await bot.AnswerCallbackQueryAsync(callbackQuery.Id, "Пользователь не найден.", cancellationToken: cancellationToken);
                }
            }
            else if (action == "info")
            {
                var users = LoadUsers();
                var user = users.FirstOrDefault(u => u.id == userId);
                if (user != null)
                {
                    var message = $"ID: {user.id}\n" +
                        $"IDs: {string.Join(", ", user.ids)}\n" +
                        $"Expires: {user.expires}\n" +
                        $"Group: {user.group}\n" +
                        $"Ban: {user.ban}\n" +
                        $"Ban Message: {user.ban_msg}\n" +
                        $"Comment: {user.comment}\n" +
                        $"Params: \n-Chatid:{user.Params.chatId} \n-Registration:{user.Params.registration}";

                    await bot.SendTextMessageAsync(chatId, message, cancellationToken: cancellationToken);
                }
                else
                {
                    await bot.SendTextMessageAsync(chatId, "Пользователь не найден.", cancellationToken: cancellationToken);
                }
            }
            else if (action == "addids")
            {
                var users = LoadUsers();
                var user = users.FirstOrDefault(u => u.id == userId);
                if (user != null)
                {
                    var response = "Введите дополнительный ID:";
                    var states = LoadState() ?? new List<StateInfo>();
                    var stateinfo = states.FirstOrDefault(s => s.chatId == chatId) ?? new StateInfo { chatId = chatId, state = "addid", id = userId };
                    stateinfo.state = "addid";
                    stateinfo.id = userId;
                    SaveState(states);
                    await bot.SendTextMessageAsync(chatId, response, cancellationToken: cancellationToken);
                }
                else
                {
                    await bot.SendTextMessageAsync(chatId, "Пользователь не найден.", cancellationToken: cancellationToken);
                }
            }
            else if (action == "extendservice")
            {
                var users = LoadUsers();
                var user = users.FirstOrDefault(u => u.id == userId);
                if (user != null)
                {
                    user.expires = DateTime.Parse(user.expires).AddMonths(1).ToString("yyyy-MM-ddTHH:mm:sszzz");
                    SaveUsers(users);
                    await bot.SendTextMessageAsync(chatId, $"Доступ продлен до {DateTime.Parse(user.expires).ToString("dd.MM.yyyy")}.", cancellationToken: cancellationToken);
                }
                else
                {
                    await bot.SendTextMessageAsync(chatId, "Пользователь не найден.", cancellationToken: cancellationToken);
                }
            }
            else if (action == "editparams")
            {
                var users = LoadUsers();
                var user = users.FirstOrDefault(u => u.id == userId);
                if (user != null)
                {
                    var inlineKeyboard = new InlineKeyboardMarkup(new[]
                    {
                        new[]
                        {
                            InlineKeyboardButton.WithCallbackData($"Torrents: {(user.Params.torrents ? "✅" : "❌")}", $"toggle:{user.id}:torrents")
                        },
                        new[]
                        {
                            InlineKeyboardButton.WithCallbackData($"VIP: {(user.Params.vip ? "✅" : "❌")}", $"toggle:{user.id}:vip")
                        }
                    });
                    await bot.SendTextMessageAsync(chatId, $"ID: {user.id}", replyMarkup: inlineKeyboard, cancellationToken: cancellationToken);
                }
                else
                {
                    await bot.SendTextMessageAsync(chatId, "Пользователь не найден.", cancellationToken: cancellationToken);
                }
            }
            else if (action == "toggle")
            {
                var param = data[2];
                var users = LoadUsers();
                var user = users.FirstOrDefault(u => u.id == userId);
                if (user != null)
                {
                    if (param == "torrents")
                    {
                        user.Params.torrents = !user.Params.torrents;
                    }
                    else if (param == "vip")
                    {
                        user.Params.vip = !user.Params.vip;
                    }
                    SaveUsers(users);

                    var inlineKeyboard = new InlineKeyboardMarkup(new[]
                    {
                        new[]
                        {
                            InlineKeyboardButton.WithCallbackData($"Torrents: {(user.Params.torrents ? "✅" : "❌")}", $"toggle:{user.id}:torrents")
                        },
                        new[]
                        {
                            InlineKeyboardButton.WithCallbackData($"VIP: {(user.Params.vip ? "✅" : "❌")}", $"toggle:{user.id}:vip")
                        }
                    });
                    await bot.EditMessageReplyMarkupAsync(chatId, callbackQuery.Message.MessageId, inlineKeyboard, cancellationToken: cancellationToken);
                }
                else
                {
                    await bot.SendTextMessageAsync(chatId, "Пользователь не найден.", cancellationToken: cancellationToken);
                }
            }
        }

        public static void SaveState(List<StateInfo> states)
        {
            try
            {
                var stateData = JsonConvert.SerializeObject(states, Formatting.Indented);
                File.WriteAllText(stateFilePath, stateData);
            }
            catch (Exception ex)
            {
                Console.WriteLine("SaveState Exception: " + ex);
            }
        }

        public static List<StateInfo> LoadState()
        {
            try
            {
                if (File.Exists(stateFilePath))
                {
                    var stateData = File.ReadAllText(stateFilePath);
                    var statelist = JsonConvert.DeserializeObject<List<StateInfo>>(stateData);
                    return statelist ?? new List<StateInfo>();
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine("LoadState Exception: " + ex);
            }
            return new List<StateInfo>();
        }

        private static List<User> LoadUsers()
        {
            try
            {
                if (File.Exists(userFilePath))
                {
                    var userData = File.ReadAllText(userFilePath);
                    var users = JsonConvert.DeserializeObject<List<User>>(userData);
                    return users ?? new List<User>();
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine("LoadUsers Exception: " + ex);
            }
            return new List<User>();
        }

        private static void SaveUsers(List<User> users)
        {
            try
            {
                var settings = new JsonSerializerSettings
                {
                    NullValueHandling = NullValueHandling.Ignore,
                    Formatting = Formatting.Indented
                };

                File.WriteAllText(userFilePath, JsonConvert.SerializeObject(users, settings));
            }
            catch (Exception ex)
            {
                Console.WriteLine("SaveUsers Exception: " + ex);
            }
        }

        public static void AddUser(string _id, List<string> _ids = null, string _expires = null, int? _group = null, bool? _ban = null, string _ban_msg = null, string _comment = null, long _chatId = 0, string _registration = null)
        {
            try
            {
                var users = LoadUsers();
                var newUser = new User
                {
                    id = _id,
                    ids = _ids ?? new List<string>(),
                    expires = _expires ?? DateTime.Now.ToString("yyyy-MM-ddTHH:mm:sszzz"),
                    group = _group,
                    ban = _ban,
                    ban_msg = _ban_msg,
                    comment = _comment,
                    Params = new Params { 
                        chatId = _chatId, 
                        registration = _registration ?? DateTime.Now.ToString("yyyy-MM-ddTHH:mm:sszzz"),
                        torrents = false,
                        vip = false}
                };

                users.Add(newUser);
                var settings = new JsonSerializerSettings
                {
                    NullValueHandling = NullValueHandling.Ignore,
                    Formatting = Formatting.Indented
                };

                File.WriteAllText(userFilePath, JsonConvert.SerializeObject(users, settings));
            }
            catch (Exception ex)
            {
                Console.WriteLine("AddUser Exception: " + ex);
            }
        }

        private static Task ErrorHandler(ITelegramBotClient botClient, Exception exception, CancellationToken cancellationToken)
        {
            var errorMessage = exception switch
            {
                ApiRequestException apiRequestException
                    => $"Telegram API Error:\n[{apiRequestException.ErrorCode}]\n{apiRequestException.Message}",
                _ => exception.ToString()
            };

            Console.WriteLine(errorMessage);
            return Task.CompletedTask;
        }
    }
}

public class User
{
    public string id { get; set; }
    public List<string> ids { get; set; }
    public string expires { get; set; }
    public int? group { get; set; }
    public bool? ban { get; set; }
    public string ban_msg { get; set; }
    public string comment { get; set; }

    [JsonProperty("params")]
    public Params Params { get; set; }
}

public class Params
{
    public long chatId { get; set; }
    public string registration { get; set; }
    public bool torrents { get; set; }
    public bool vip { get; set; }
}

public class StateInfo
{
    public long chatId { get; set; }
    public string state { get; set; }
    public string id { get; set; }
}